/**
 * ========================================
 * ジョイスティック(SW付き)のバージョン
 * ========================================
 *
 * 【概要】
 * アナログジョイスティック(X軸、Y軸、SWボタン)の状態を読み取り、
 * シリアル通信でPCへ送信するプログラムです。
 *
 * 【ハードウェア構成】
 * - X軸: アナログピン A0 に接続
 * - Y軸: アナログピン A1 に接続
 * - SWボタン: デジタルピン 5 に接続(INPUT_PULLUP使用)
 *
 * 【通信仕様】
 * - ボーレート: 115200 bps
 * - データ形式: key:value,key:value 形式
 * - 送信間隔: 100ms(0.1秒)ごと
 * - 送信例: "joystick_x:512,joystick_y:1023,joystick_sw:1"
 *
 * 【データ範囲】
 * - X軸、Y軸: 0〜1023 (10bit アナログ値)
 * - SWボタン: 0 = 押下中、1 = 未押下
 */

// ========================================
// ピン番号の定義
// ========================================
const int VRx_Pin = A0;  // X軸のアナログ入力ピン
const int VRy_Pin = A1;  // Y軸のアナログ入力ピン
const int SW_Pin = 5;    // SWボタンのデジタル入力ピン

// ========================================
// 送信間隔の設定
// ========================================
const unsigned long INTERVAL_MS = 100; // データ送信間隔(ミリ秒) = 0.1秒

// ========================================
// グローバル変数
// ========================================
unsigned long lastSendTime = 0;  // 前回データを送信した時刻(ミリ秒)

/**
 * ========================================
 * 初期設定
 * ========================================
 * Arduino起動時に1度だけ実行される関数です。
 * シリアル通信の開始とピンモードの設定を行います。
 */
void setup() {
  // シリアル通信を開始(ボーレート: 115200bps)
  Serial.begin(115200);

  // SWボタンのピンを入力プルアップモードに設定
  // INPUT_PULLUP: 内部プルアップ抵抗を有効化
  //   - ボタン未押下時: HIGH (約5V)
  //   - ボタン押下時:   LOW  (0V = GND)
  pinMode(SW_Pin, INPUT_PULLUP);
}

/**
 * ========================================
 * メインループ
 * ========================================
 * setup()の後、繰り返し実行される関数です。
 * 一定間隔でジョイスティックの状態をシリアル送信します。
 */
void loop() {
  unsigned long now = millis();  // 現在の経過時間(ミリ秒)を取得

  // 前回の送信から INTERVAL_MS 以上経過していれば送信
  if (now - lastSendTime >= INTERVAL_MS) {
    lastSendTime = now;         // 送信時刻を更新
    sendJoystickState();        // ジョイスティック状態を送信
  }
}

/**
 * ========================================
 * ジョイスティック状態の送信
 * ========================================
 * X軸、Y軸、SWボタンの状態を読み取り、
 * key:value 形式でシリアル送信します。
 *
 * 【送信フォーマット】
 * joystick_x:<X値>,joystick_y:<Y値>,joystick_sw:<SW値>
 *
 * 例: joystick_x:512,joystick_y:1023,joystick_sw:1
 */
void sendJoystickState() {
  // ========================================
  // センサー値の読み取り
  // ========================================

  // X軸の値を取得 (0〜1023 の範囲)
  int X_Pos  = analogRead(VRx_Pin);

  // Y軸の値を取得 (0〜1023 の範囲)
  int Y_Pos  = analogRead(VRy_Pin);

  // SWボタンの状態を取得
  // HIGH: ボタン未押下 (プルアップにより約5V)
  // LOW:  ボタン押下   (GNDに接続されて0V)
  int SW_Raw = digitalRead(SW_Pin);

  // ========================================
  // SWボタンの状態を変換
  // ========================================
  // digitalRead() の戻り値 (HIGH/LOW) を、
  // アプリケーション側で扱いやすい形式 (0/1) に変換
  //   - ボタン押下時:   SW_Raw = LOW  → SW_State = 0
  //   - ボタン未押下時: SW_Raw = HIGH → SW_State = 1
  int SW_State = (SW_Raw == LOW) ? 0 : 1;

  // ========================================
  // シリアル送信
  // ========================================
  // key:value 形式でデータを送信
  // カンマ区切りで複数のデータを1行にまとめる

  Serial.print("joystick_x:");   // X軸のキー
  Serial.print(X_Pos);            // X軸の値
  Serial.print(",joystick_y:");  // Y軸のキー
  Serial.print(Y_Pos);            // Y軸の値
  Serial.print(",joystick_sw:"); // SWボタンのキー
  Serial.println(SW_State);       // SWボタンの値(改行付き)

  // 送信例: "joystick_x:512,joystick_y:1023,joystick_sw:1\n"
}
