/**
 * ========================================
 * タクトスイッチのみのバージョン
 * ========================================
 *
 * 【概要】
 * タクトスイッチの状態を読み取り、
 * シリアル通信でPCへ送信するプログラムです。
 *
 * 【ハードウェア構成】
 * - SW1: デジタルピン 2 に接続(INPUT_PULLUP使用)
 *
 * 【通信仕様】
 * - ボーレート: 115200 bps
 * - データ形式: key:value 形式
 * - 送信間隔: 100ms(0.1秒)ごと
 * - 送信例: "sw1:0" または "sw1:1"
 *
 * 【データ範囲】
 * - SW1: 0 = 押下中、1 = 未押下
 */

// ========================================
// ピン番号の定義
// ========================================
const int SW1_PIN = 2;  // タクトスイッチのデジタル入力ピン

// ========================================
// 送信間隔の設定
// ========================================
const unsigned long INTERVAL_MS = 100; // データ送信間隔(ミリ秒) = 0.1秒

// ========================================
// グローバル変数
// ========================================
unsigned long lastSendTime = 0;  // 前回データを送信した時刻(ミリ秒)

/**
 * ========================================
 * 初期設定
 * ========================================
 * Arduino起動時に1度だけ実行される関数です。
 * シリアル通信の開始とピンモードの設定を行います。
 */
void setup() {
  // タクトスイッチのピンを入力プルアップモードに設定
  // INPUT_PULLUP: 内部プルアップ抵抗を有効化
  //   - スイッチ未押下時: HIGH (約5V)
  //   - スイッチ押下時:   LOW  (0V = GND)
  pinMode(SW1_PIN, INPUT_PULLUP);

  // シリアル通信を開始(ボーレート: 115200bps)
  Serial.begin(115200);
}

/**
 * ========================================
 * メインループ
 * ========================================
 * setup()の後、繰り返し実行される関数です。
 * 一定間隔でタクトスイッチの状態をシリアル送信します。
 */
void loop() {
  unsigned long now = millis();  // 現在の経過時間(ミリ秒)を取得

  // 前回の送信から INTERVAL_MS 以上経過していれば送信
  if (now - lastSendTime >= INTERVAL_MS) {
    lastSendTime = now;         // 送信時刻を更新
    sendSwitchState();          // タクトスイッチ状態を送信
  }
}

/**
 * ========================================
 * タクトスイッチ状態の送信
 * ========================================
 * タクトスイッチの状態を読み取り、
 * key:value 形式でシリアル送信します。
 *
 * 【送信フォーマット】
 * sw1:<SW値>
 *
 * 例: sw1:0 (押下中) または sw1:1 (未押下)
 */
void sendSwitchState() {
  // ========================================
  // スイッチ状態の読み取り
  // ========================================

  // タクトスイッチの状態を取得
  // HIGH: スイッチ未押下 (プルアップにより約5V)
  // LOW:  スイッチ押下   (GNDに接続されて0V)
  int raw1 = digitalRead(SW1_PIN);

  // ========================================
  // スイッチ状態の変換
  // ========================================
  // digitalRead() の戻り値 (HIGH/LOW) を、
  // アプリケーション側で扱いやすい形式 (0/1) に変換
  //   - スイッチ押下時:   raw1 = LOW  → sw1 = 0
  //   - スイッチ未押下時: raw1 = HIGH → sw1 = 1
  int sw1 = (raw1 == LOW) ? 0 : 1;

  // ========================================
  // シリアル送信
  // ========================================
  // key:value 形式でデータを送信

  Serial.print("sw1:");     // スイッチのキー
  Serial.println(sw1);      // スイッチの値(改行付き)

  // 送信例: "sw1:0\n" (押下中) または "sw1:1\n" (未押下)
}
