/**
 * app.js
 * ====================================
 * Arduino I/O Module（arduino-io.js）を利用した
 * アプリケーション側の制御・UI連携を行うスクリプトです。
 *
 * 本ファイルでは以下を担当します。
 * - Arduino との接続／切断操作（UI操作）
 * - Arduino からの入力イベントの受信
 * - 受信データを画面表示やアプリケーションロジックへ反映
 *
 * ※ シリアル通信そのものの処理は arduino-io.js 側に委譲しています。
 */


/* =====================================================
   UI要素の取得
   -----------------------------------------------------
   HTML 側に用意された要素を取得し、
   接続状態やスイッチ状態の表示に使用します。
===================================================== */

// Arduino 接続／切断ボタン
const connectBtn = document.getElementById('connectBtn');

// 接続状態表示用テキスト
const connectStatus = document.getElementById('connectStatus');

// Arduino スイッチ状態表示用テキスト
const buttonState = document.getElementById('buttonState');


/* =====================================================
   接続ボタンのイベント処理
   -----------------------------------------------------
   ボタンを押すたびに、
   - 未接続 → 接続
   - 接続中 → 切断
   を切り替えます。
===================================================== */

async function handleArduinoConnect() {

    // 現在の接続状態を arduino-io.js 側に問い合わせる
    if (!Arduino.isConnected()) {

        // 未接続の場合：Arduino に接続する
        const success = await Arduino.connect();

        if (success) {
            // 接続成功時のUI更新
            connectStatus.textContent = '接続中';
            connectBtn.textContent = 'Disconnect';
        } else {
            // 接続失敗時のUI更新
            connectStatus.textContent = '接続失敗';
        }

    } else {

        // 接続中の場合：Arduino との接続を切断する
        await Arduino.disconnect();

        // 切断後のUI更新
        connectStatus.textContent = '未接続';
        connectBtn.textContent = 'Connect';
    }
}

connectBtn.addEventListener('click', handleArduinoConnect);


/* =====================================================
   Arduino 入力イベントの受信処理
   -----------------------------------------------------
   arduino-io.js から発火される CustomEvent 'arduino-data' を受信します。
 * e.detail には、Arduino 側から送信された
 * key:value 形式のデータがオブジェクトとして格納されます。
===================================================== */

function handleArduinoData(e) {

    // 解析済みの Arduino 入力データ
    const data = e.detail;

    console.log(data); // デバッグ用：受信データ確認

    // sw1 キーが含まれている場合のみ処理する
    if (data.sw1 !== undefined) {

        const button = data.sw1;

        // Arduino 側のスイッチ状態を画面表示に反映
        // ※ 例：0 = 押されている / 1 = 押されていない
        buttonState.textContent = button === 0 ? 'ON' : 'OFF';
    }
}

window.addEventListener('arduino-data', handleArduinoData);