/**
 * Arduino I/O Module
 * 2025.12.29  @FOURIER INC.
 * ====================================
 * Web Serial API を使用して、Arduino とブラウザ間で
 * シリアル通信（入出力）を行うための汎用モジュールです。
 *
 * Arduino から送信されるテキストデータ（1行単位）を受信・解析し、
 * CustomEvent（'arduino-input'）としてアプリケーションへ通知します。
 *
 * ====================================================
 * 使い方（例）
 * ====================================================
 *
 * 【1】HTML に読み込む（DOM構築後に実行されるよう defer 推奨）
 *   <script src="arduino-io.js" defer></script>
 *
 * 【2】ユーザー操作をきっかけに接続する（Web Serial API の仕様）
 *   document.querySelector('#connectBtn').addEventListener('click', async () => {
 *     await Arduino.connect();
 *   });
 *
 * 【3】Arduino からのデータを受信する
 *   window.addEventListener('arduino-data', (e) => {
 *     console.log(e.detail); // { sw1: 1, sw2: 0, joyX: 512 } のようなオブジェクト
 *   });
 *
 * ====================================================
 * Arduino 側の送信フォーマット
 * ====================================================
 * Arduino は 1 行ごと（改行 \n 区切り）に送信してください。
 * 複数データは「key:value」をカンマ区切りで記述します。
 *
 * 送信例：
 *   sw1:1,sw2:0,joyX:512
 *
 * 受信側（JS）では以下のように解釈されます：
 *   {
 *     sw1: 1,
 *     sw2: 0,
 *     joyX: 512
 *   }
 *
 * ====================================================
 * 注意点
 * ====================================================
 * - Web Serial API 対応ブラウザ（主に Chrome / Edge）が必要です
 * - HTTPS または localhost 環境でのみ動作します
 * - 大量データの高速送信はバッファオーバーフローに注意してください
 * - 本コードはサンプルとして提供されるものであり、セキュリティ対策および動作保証は行っていません
 */

// =====================================================
// 内部状態（通信のための保持変数）
// =====================================================

// シリアルポート本体（接続先）
let port = null;

// 読み取りループを継続するかどうかのフラグ
let keepReading = false;

// 接続状態フラグ（外部から参照用）
let isConnected = false;

// 受信データを一時的に蓄積するバッファ
// 受信が改行単位で来ない場合があるため、改行が来るまで貯める
let serialBuffer = '';

// 受信用 reader（ReadableStreamDefaultReader<string>）
let reader = null;

// 送信用 writer（WritableStreamDefaultWriter）
let writer = null;

// port.readable.pipeTo(...) の戻り値（Promise）
// 切断時に await してストリームを安全にクローズするため保持
let readableStreamClosed = null;

// テキストエンコーダー（文字列をバイト配列に変換）
const textEncoder = new TextEncoder();


// =====================================================
// Arduino 公開API（外部から使うインターフェース）
// =====================================================
const Arduino = {
    /**
     * Arduino に接続する
     * - Web Serial API の仕様上、ユーザー操作（クリック等）から呼び出す必要があります
     * @returns {Promise<boolean>} 接続成功: true / 失敗: false
     */
    async connect() {
        try {
            // Web Serial API 対応チェック
            if (!('serial' in navigator)) {
                throw new Error('このブラウザは Web Serial API に対応していません');
            }

            // ユーザーに接続先ポートを選ばせる（ブラウザのダイアログが出る）
            port = await navigator.serial.requestPort();

            // シリアルポートを開く（Arduino 側と同じボーレートにする）
            await port.open({ baudRate: 115200 });

            // -------------------------------------------------
            // 受信パイプライン構築：
            //   port.readable(バイナリ) → TextDecoderStream(文字列) → reader.read()
            // -------------------------------------------------
            const textDecoderStream = new TextDecoderStream();

            // pipeTo の Promise は切断時の後始末に必要なので保持しておく
            readableStreamClosed = port.readable.pipeTo(textDecoderStream.writable);

            // 文字列ストリームから読む reader を取得
            reader = textDecoderStream.readable.getReader();

            // -------------------------------------------------
            // 送信用 writer を取得
            // -------------------------------------------------
            writer = port.writable.getWriter();

            // 状態更新
            isConnected = true;
            keepReading = true;

            console.log("Arduinoと接続しました");

            // 受信ループ開始（非同期で回し続ける）
            readLoop().catch(err => {
                console.error('readLoop error:', err);
            });

            return true;
        } catch (err) {
            console.error('Arduinoと接続に失敗しました:', err);
            return false;
        }
    },

    /**
     * Arduino との接続を切断する
     * - reader / writer / port / pipeTo を順番に片付ける
     */
    async disconnect() {
        // 読み取りループを停止
        keepReading = false;
        isConnected = false;

        // reader が存在する場合はキャンセルして終了させる
        if (reader) {
            try {
                await reader.cancel();
            } catch (error) {
                console.warn('Reader cancel error:', error);
            }
            reader = null;
        }

        // pipeTo の完了を待つ（すでに閉じている場合は例外になることがあるため握りつぶす）
        if (readableStreamClosed) {
            try {
                await readableStreamClosed;
            } catch (error) {
                // Stream already closed or aborted
            }
            readableStreamClosed = null;
        }

        // writer のロックを解放
        if (writer) {
            try {
                writer.releaseLock();
            } catch (error) {
                console.warn('Writer release error:', error);
            }
            writer = null;
        }

        // ポートを閉じる
        if (port) {
            try {
                await port.close();
            } catch (error) {
                console.warn('Port close error:', error);
            }
            port = null;
        }

        // バッファは任意。再接続時に混ざるのが嫌ならクリアする
        serialBuffer = '';

        console.log("切断しました");
    },

    /**
     * 接続状態を取得する
     * @returns {boolean}
     */
    isConnected() {
        return isConnected;
    },

    /**
     * Arduino にコマンドを送信する
     * @param {string} command - 送信するコマンド文字列
     * @returns {Promise<boolean>} 送信成功: true / 失敗: false
     */
    async sendCommand(command) {
        if (!isConnected || !writer) {
            console.error('Arduino未接続、または writer が利用できません');
            return false;
        }

        try {
            // コマンドに改行を付けてバイト配列に変換
            const data = textEncoder.encode(command + '\n');

            // writer に書き込む
            await writer.write(data);

            console.log('コマンド送信:', command);
            return true;
        } catch (err) {
            console.error('コマンド送信エラー:', err);
            return false;
        }
    }
};


// =====================================================
// 受信処理
// =====================================================

/**
 * Arduino からのデータを読み続けるループ
 * - keepReading が false になるか、reader が終了すると停止します
 * - 受信は任意のチャンクで来るため、改行単位になるようバッファリングします
 */
async function readLoop() {
    while (keepReading && reader) {
        try {
            // reader.read() は { value: string, done: boolean } を返す
            const { value, done } = await reader.read();

            // ストリームが終了した場合
            if (done) break;

            if (value) {
                // 受信した文字列を一旦バッファに追加
                serialBuffer += value;

                // 改行単位で分割（最後の要素は未完了行の可能性あり）
                const lines = serialBuffer.split('\n');

                // 最後の未完了行は次回受信まで保持
                serialBuffer = lines.pop() || '';

                // 完成した行のみパースする
                for (const line of lines) {
                    parseArduinoData(line.trim());
                }
            }
        } catch (error) {
            // keepReading が true のときに落ちたならログに出す（デバッグしやすくする）
            if (keepReading) {
                console.error('読み取りエラー:', error);
            }
            break;
        }
    }
}


// =====================================================
// パース処理（1行 → オブジェクト）
// =====================================================

/**
 * Arduino から受信した 1 行分のテキストデータを解析し、
 * "key:value,key:value" 形式を { key: number } のオブジェクトへ変換します。
 *
 * @param {string} dataString - 1行分の文字列（例: "sw1:1,sw2:0,joyX:512"）
 */
function parseArduinoData(dataString) {
    if (!dataString) return;

    const data = {};

    // 例: "sw1:1,sw2:0,joyX:512" をカンマで分割してペアにする
    const pairs = dataString.split(',');

    for (const pair of pairs) {
        // "sw1:1" → ["sw1", "1"]
        const [key, value] = pair.split(':');

        // key があり、value が存在する場合のみ処理
        if (key && value !== undefined) {
            const k = key.trim();
            const n = parseInt(value.trim(), 10);

            // 数値として解釈できる場合のみ採用（NaN なら入れない）
            if (!Number.isNaN(n)) {
                data[k] = n;
            }
        }
    }

    // -------------------------------------------------
    // パースしたデータをイベントとして通知する
    // - イベント名: 'arduino-data'
    // - e.detail : 解析済みオブジェクト（例: { sw1: 1, joyX: 512 }）
    // -------------------------------------------------
    if (Object.keys(data).length > 0) {
        window.dispatchEvent(new CustomEvent('arduino-data', {
            detail: data
        }));
    }
}